#ifndef __CBNODEBASE_H__
#define __CBNODEBASE_H__
/** \file CBNodeBase.h
 *  \brief CBNodeBase base class header
 *  \details File contain CBNodeBase definition.
 *  \warning Don't modify this file
 *	\author Pety 2012
 *	$Id: CBNodeBase.h 1 2012-10-15 13:25:08Z petyovsky $
 */

#include <iostream>
#include <sstream>
#include <stdexcept>
#include <typeinfo>
#include "demagle.h"
#include "check.h"

#ifdef WIN32
	#pragma warning(disable: 4355) // Due disable warning C4355: 'this' : used in base member initializer list
#endif

/** \brief CBNodeBase base class
 *  \details Definition of CBNodeBase base class. There is defined all common methods and attributes.
 */
class CBNodeBase
	{
	static unsigned iTotalCounter;	///< Summary count of all created instances
	static int iLivingCounter;	///< Counter of actually existing instances
	mutable CBNodeBase *iLeft;	///< Pointer to the left node (\em intrusive attribute)
	mutable CBNodeBase *iRight;	///< Pointer to the right node (\em intrusive attribute)
	mutable int iExtraInt;      ///< Auxiliary \c int value is useful for some projects only: multiset(for same value nodes counter], priority queue[for priority value] (\em intrusive attribute)
	const unsigned iIndex;		///< Constant value for simple runtime instance identification and debuging

//c'tors
	CBNodeBase(const CBNodeBase& aBNode); ///< Defined like a \c private for disallowing compiler implicit generation of copy c'tor

public:
/** \brief Implicit c'tor
 *  \details Pointer atributes are initialised to the \c this value others atributes contains zero.
 */
	CBNodeBase(): iLeft(this), iRight(this), iExtraInt(0), iIndex(iTotalCounter++)
		{iLivingCounter++;}
	
/** \brief Conversion c'tor
 *  \param[in]	aLeft	Pointer to the left linked CBNodeBase
 *  \param[in]	aRight	Pointer to the right linked CBNodeBase
 *  \param[in]	aExtraInt	Auxiliary \c int value needed for project: multiset, priority queue...
 *  \attention Method generate \c std::runtime_error exception if parameters \p aLeft or \p aRight contain \c nullptr value.
 *  \note Using of auxiliary \p ExtraInt value is useful for some projects only: multiset[same values counter], priority queue[priority]
 */
	CBNodeBase(CBNodeBase *aLeft, CBNodeBase *aRight, int aExtraInt = 0): iExtraInt(aExtraInt), iIndex(iTotalCounter++)
		{SetLeft(aLeft); SetRight(aRight); iLivingCounter++;}

//d'tor
/** \brief Virtual d'tor
 *  \attention Method check and generate \c std::runtime_error exception if instance is part of the any list! So both pointers \p iLeft and \p iRight must point to the \c this value before you call d'tor.
 *  \note Generating exceptions from d'tors is problematic practice (for example during early development process), because can generate multiple exceptions in the same moment. So, throwing of \c std::runtime_error exception can be supress be undefining preprocessor symbol \c CBNODE_DTOR_EXCEPTIONS, but error message is always reported to the \c std::cerr.
 */
	virtual ~CBNodeBase()
		{
		iLivingCounter--;
		if( (Left() != this) || (Right() != this))
			{
			std::ostringstream oss;
			oss << "Instance of class: \'" << DM(typeid(*this).name()) << '#' << Index() << "\' is destructed with connected list from: ";
			if(Left() != this)
				{
				oss << "left";
				if(Right() != this)
					oss << " and right";
				}
			else
				oss << "right";
			oss << "!!";
			
#ifdef CBNODE_DTOR_EXCEPTIONS
			throw std::runtime_error(oss.str());
#else
			std::cerr << std::endl << "ERROR: " << oss.str() << std::endl << '\a';
#endif
			}
		}

// intrusive setters/getters
/** \brief Left pointer setter
 *  \param[in]	aLeft	Pointer to the new left linked CBNodeBase
 *  \return Previous \p left pointer value
 *  \attention Method generate \c std::runtime_error exception if parameters \p aLeft contain \c nullptr value.
 */
	CBNodeBase* SetLeft(CBNodeBase *aLeft) const
		{
		if(!aLeft)
			{
			std::ostringstream oss; oss << "Instance of class: \'" << DM(typeid(*this).name()) << '#' << Index() << "\' is connected from left to NULL!!";
			throw std::runtime_error(oss.str());
			}
		CBNodeBase *oldleft(iLeft);
		iLeft = aLeft;
		return(oldleft);
		}
/** \brief Left pointer getter
 *  \return Actual \p left pointer value
 */
	CBNodeBase* Left() const
		{return(iLeft);}
	
/** \brief Right pointer setter
 *  \param[in]	aRight	Pointer to the new right linked CBNodeBase
 *  \return Previous \p right pointer value
 *  \attention Method generate \c std::runtime_error exception if parameters \p aRight contain \c nullptr value.
 */
	CBNodeBase* SetRight(CBNodeBase *aRight) const
		{
		if(!aRight)
			{
			std::ostringstream oss; oss << "Instance of class: \'" << DM(typeid(*this).name()) << '#' << Index() << "\' is connected from right to NULL!!";
			throw std::runtime_error(oss.str());
			}

		CBNodeBase *oldright(iRight);
		iRight = aRight;
		return(oldright);
		}
/** \brief Right pointer getter
 *  \return Actual \p right pointer value
 */
	CBNodeBase* Right() const
		{return(iRight);}
	
/** \brief Combined Left and Right pointers setter
 *  \param[in]	aLeft	Pointer to the new left linked CBNodeBase
 *  \param[in]	aRight	Pointer to the new right linked CBNodeBase
 *  \attention Method generate \c std::runtime_error exception if parameters \p aLeft or \p aRight contain \c nullptr value.
 */
	void SetLeftRight(CBNodeBase *aLeft, CBNodeBase *aRight) const
		{
		SetLeft(aLeft); SetRight(aRight);
		}
	
// intrusive setter/getter, useful for some projects only: multiset, priority queue
/** \brief Auxiliary ExtraInt value setter
 *  \param[in]	aExtraInt	New ExtraInt value
 *  \note Using of auxiliary \p ExtraInt value is useful for some projects only: multiset[same values counter], priority queue[priority]
 */
	void SetExtraInt(const int aExtraInt) const
		{iExtraInt = aExtraInt;}
	
/** \brief Auxiliary ExtraInt value getter
 *  \return Actual \p ExtraInt value
 *  \note Using of auxiliary \p ExtraInt value is useful for some projects only: multiset[same values counter], priority queue[priority]
 */
	int ExtraInt() const
		{return(iExtraInt);}

// counters getters
/** \brief Getter for actual count of all created instances
 *  \return Actual count of all created instances
 */
	static unsigned Total()
		{return(iTotalCounter);}
	
/** \brief Getter for count of actually existing instances
 *  \return Count of actually existing instances
 */
	static int Living()
		{return(iLivingCounter);}
	
/** \brief Index getter
 *  \return Unicate instance index
 */
	unsigned Index() const
		{return(iIndex);}

	}; /* class CBNodeBase */

#endif /* __CBNODEBASE_H__ */
