#ifndef __CBNODE_TWEEKDAY_H__
#define __CBNODE_TWEEKDAY_H__
/** \file CBNode_TWeekDay.h
 *  \brief CBNode_TWeekDay class header
 *  \details File contain CBNode_TWeekDay::CBNode definition.
 *  \warning Don't modify this file
 *	\author Pety 2012
 *	$Id: CBNode_TWeekDay.h 1 2011-10-17 13:36:18Z petyovsky 
 */

#include "CBNodeBase.h"
#include "check.h"

/** \brief Namespace for encapsulating of \c TWeekDay variant of CBNode class
 *  \details For selecting this variant of CBNode class uncomment \c using \c namespace section in the CBNode.h
 */
namespace CBNode_TWeekDay
	{
	/** \brief Basic definition of enumeration type for representing day of week.
	 */
	enum TWeekDay
		{EMonday = 0, ETuesday, EWednesday, EThursday, EFriday, ESaturday, ESunday};

	// Definition of support functions a global operators for type enum TWeekDay

	/** \brief Conversion and range checking function. Convert \c unsigned value to the TWeekDay
	 *  \param[in]	aDayNum		Number of day
	 *  \return Return TWeekDay value
	 *  \attention Method generate \c std::range_error exception if parameters \p aDayNum is out of numbers of the day
	 */
	TWeekDay CheckingRangeWithException(const unsigned aDayNum);
	
	/** \brief Output to the stream operator. (\em serialization)
	 *  \param[in]	aOStream	Output stream
	 *  \param[in]	aWeekDay	Serialized instantion of TWeekDay
	 *  \return Return \c std::ostream with serialized aWeekDay
	 */
	std::ostream& operator<<(std::ostream& aOStream, const TWeekDay& aWeekDay);
	
	/** \brief Input from the stream operator. (\em deserialization)
	 *  \param[in]	aIStream	Input stream
	 *  \param[out]	aWeekDay	Place for deserialized instantion of TWeekDay
	 *  \return Return rest of \c std::istream
	 *  \attention Method generate \c std::range_error exception if stream doesn't contains correct name of the day
	 */
	std::istream& operator>>(std::istream& aIStream, TWeekDay& aWeekDay);
	
	
	/** \brief CBNode class (\c TWeekDay variant)
	 *  \details Definition of CBNode class (\c TWeekDay variant). There is defined all common methods and attributes.
	 */
	class CBNode: public CBNodeBase
		{				
		enum TWeekDay iValue;	///< Encapsulated \c enum TWeekDay value

	public:
	// c'tors
	/** \brief Implicit c'tor
	 *  \details Value attributes is set to \c EMonday. Inherited attributes are initialised by CBNodeBase implicit c'tor.
	 */
		CBNode(): CBNodeBase(), iValue(CheckingRangeWithException(EMonday))
			{}
			
	/** \brief Conversion c'tor
	 *  \details Pointer attributes are initialised to the \c this value.
   	 *  \param[in]	aValue New TWeekDay value
	 *  \param[in]	aExtraInt	Auxiliary \c int value
	 *  \attention Method generate \c std::range_error exception if parameters \p aValue is out of the \c enum TWeekDay range.
	 *  \note Using of auxiliary \p ExtraInt value is useful for some projects only: multiset[same values counter], priority queue[priority]
	 */
		CBNode(const enum TWeekDay aValue, int aExtraInt = 0): CBNodeBase(this, this, aExtraInt), iValue(CheckingRangeWithException(aValue))
			{}
	/** \brief Conversion c'tor
   	 *  \param[in]	aValue New TWeekDay value
	 *  \param[in]	aLeft	Pointer to the left linked CBNodeBase
	 *  \param[in]	aRight	Pointer to the right linked CBNodeBase
	 *  \param[in]	aExtraInt	Auxiliary \c int value
	 *  \attention Method generate \c std::runtime_error exception if parameters \p aLeft or \p aRight contain \c nullptr value.
	 *  \attention Method generate \c std::range_error exception if parameters \p aValue is out of the \c enum TWeekDay range.
	 *  \note Using of auxiliary \p ExtraInt value is useful for some projects only: multiset[same values counter], priority queue[priority]
	 */
		CBNode(const enum TWeekDay aValue, CBNode *aLeft, CBNode *aRight, int aExtraInt = 0): CBNodeBase(aLeft, aRight, aExtraInt), iValue(CheckingRangeWithException(aValue))
			{}
			
	/** \brief Copy c'tor
	 *  \details Create new instance by copying only \p iValue parameter. All intrusive attributes (pointers,ExtraInt) are set to the their default values.
	 *  \param[in]	aBNode	Original instance for copying
	 */
		CBNode(const CBNode& aBNode): CBNodeBase(), iValue(aBNode.Value())
			{}
			
	/** \brief String conversion c'tor
	 *  \details Create new instance from Value in the string. Pointer attributes are initialised to the \c this value.
	 *  \param[in] aStr Plain C string with string value convertable to the TWeekDay value
	 *  \param[in]	aExtraInt	Auxiliary \c int value
	 *  \note Using of auxiliary \p ExtraInt value is useful for some projects only: multiset[same values counter], priority queue[priority]
	 */
		CBNode(const char* aStr, int aExtraInt = 0): CBNodeBase(this, this, aExtraInt), iValue(EMonday)
			{std::istringstream iss(aStr, std::istringstream::in); iss >> iValue;}

	/** \brief String conversion c'tor
	 *  \details Create new instance from Value in the string.
	 *  \param[in] aStr Plain C string with string value convertable to the TWeekDay value
	 *  \param[in]	aLeft	Pointer to the left linked CBNodeBase
	 *  \param[in]	aRight	Pointer to the right linked CBNodeBase
	 *  \param[in]	aExtraInt	Auxiliary \c int value
	 *  \attention Method generate \c std::runtime_error exception if parameters \p aLeft or \p aRight contain \c nullptr value.
	 *  \note Using of auxiliary \p ExtraInt value is useful for some projects only: multiset[same values counter], priority queue[priority]
	 */			
		CBNode(const char* aStr, CBNode *aLeft, CBNode *aRight, int aExtraInt = 0): CBNodeBase(aLeft, aRight, aExtraInt), iValue(EMonday)
			{std::istringstream iss(aStr, std::istringstream::in); iss >> iValue;}
		
	// d'tor
	/** \brief Virtual d'tor
	 */			
		virtual ~CBNode()
			{iValue = EMonday;}

	// setter/getter
	/** \brief Value setter
	 *  \param[in]	aValue	New Value
	 *  \attention Method generate \c std::range_error exception if parameters \p aValue is out of the \c enum TWeekDay range.
	 */
		void SetValue(const TWeekDay aValue)
			{iValue = CheckingRangeWithException(unsigned(aValue));}
			
	/** \brief Value getter
	 *  \return Actual \c bool \p Value
	 */
		TWeekDay Value() const
			{return(iValue);}

	// operators
	/** \brief Complement operator
	 *  \return CBNode instance with complemented attribute Value. All intrusive attributes (pointers,ExtraInt) are set to the their default values.
	 */			
		CBNode operator-() const
			{unsigned tmp = unsigned(ESunday) - unsigned(Value());  return(CBNode(TWeekDay(tmp)));}
			
	/** \brief Assigment operator
	 *  \return CBNode instance with copied attribute Value. All intrusive attributes (pointers,ExtraInt) has previous values.
	 */
		CBNode& operator=(const CBNode& aBNode)
			{SetValue(aBNode.Value());return(*this);}
			
	/** \brief Comparing by Value operator
	 *  \return Return \c bool result of comparation
	 */			
		bool operator==(const CBNode& aBNode) const
			{return(iValue == aBNode.Value());}
			
	/** \brief Comparing by Value operator
	 *  \return Return \c bool result of comparation
	 */
		bool operator!=(const CBNode& aBNode) const
			{return(iValue != aBNode.Value());}
			
	/** \brief Comparing by Value operator
	 *  \return Return \c bool result of comparation
	 */
		bool operator<(const CBNode& aBNode) const
			{return(iValue < aBNode.Value());}
			
	/** \brief Comparing by Value operator
	 *  \return Return \c bool result of comparation
	 */
		bool operator>(const CBNode& aBNode) const
			{return(iValue > aBNode.Value());}

	/** \brief Output to the stream operator. (\em serialization)
	 *  \param[in]	aOStream	Output stream
	 *  \param[in]	aBNode	Serialized instantion of CBNode
	 *  \return Return \c std::ostream with serialized Value
	 */
		friend std::ostream& operator<<(std::ostream& aOStream, const CBNode& aBNode)
			{aOStream << aBNode.Value(); return(aOStream);}
				
	/** \brief Input from the stream operator. (\em deserialization)
	 *  \param[in]	aIStream	Input stream
	 *  \param[out]	aBNode	Place for deserialized instantion of CBNode
	 *  \return Return rest of \c std::istream
	 */
		friend std::istream& operator>>(std::istream& aIStream, CBNode& aBNode)
			{aIStream >> aBNode.iValue; return(aIStream);}

#ifdef CBNODE_TEST_VALUES
		// constants for testing CBNode functionality
	/** \brief First test value
	 *  \return Return TWeekDay value (EMonday)
	 *  \note Useful for automated testing of CBNode functionality
	 */				
		static TWeekDay TestValue0()
			{return(EMonday);}
				
	/** \brief First test string value
	 *  \return Return string with TWeekDay value (EMonday)
	 *  \note Useful for automated testing of CBNode functionality
	 */
		static const char* TestStringValue0()
			{return("(Monday)");}

	/** \brief Second test value
	 *  \return Return TWeekDay value (ETuesday)
	 *  \note Useful for automated testing of CBNode functionality
	 */				
		static TWeekDay TestValue1()
			{return(ETuesday);}

	/** \brief Second test string value
	 *  \return Return string with TWeekDay value (ETuesday)
	 *  \note Useful for automated testing of CBNode functionality
	 */
		static const char* TestStringValue1()
			{return("(Tuesday)");}
				
#endif /* CBNODE_TEST_VALUES */

		}; /* class CBNode */

	} /* namespace CBNode_TWeekDay */

#endif /* __CBNODE_TWEEKDAY_H__ */
