#ifndef __CBNODE_BOOL_H__
#define __CBNODE_BOOL_H__
/** \file CBNode_bool.h
 *  \brief CBNode_bool class header
 *  \details File contain CBNode_bool::CBNode definition.
 *  \warning Don't modify this file
 *	\author Pety 2012
 *	$Id: CBNode_bool.h 1 2012-10-15 13:25:08Z petyovsky $
 */

#include "CBNodeBase.h"
#include "check.h"

/** \brief Namespace for encapsulating of \c bool variant of CBNode class
 *  \details For selecting this variant of CBNode class uncomment \c using \c namespace section in the CBNode.h
 */
namespace CBNode_bool
	{
	/** \brief CBNode class (\c bool variant)
	 *  \details Definition of CBNode class (\c bool variant). There is defined all common methods and attributes.
	 */
	class CBNode: public CBNodeBase
		{
		bool iValue; ///< Encapsulated \c bool value

	public:
	// c'tors
	/** \brief Implicit c'tor
	 *  \details Value attributes is set to \c false. Inherited attributes are initialised by CBNodeBase implicit c'tor. 
	 */
		CBNode(): CBNodeBase(), iValue(false)
			{}
		
	/** \brief Conversion c'tor
	 *  \details Pointer attributes are initialised to the \c this value.
   	 *  \param[in]	aValue New encapsulated \c bool Value
	 *  \param[in]	aExtraInt	Auxiliary \c int value
	 *  \note Using of auxiliary \p ExtraInt value is useful for some projects only: multiset[same values counter], priority queue[priority]
	 */
		CBNode(const bool aValue, int aExtraInt = 0): CBNodeBase(this, this, aExtraInt), iValue(aValue)
			{}
		
	/** \brief Conversion c'tor
   	 *  \param[in]	aValue New encapsulated \c bool Value
	 *  \param[in]	aLeft	Pointer to the left linked CBNodeBase
	 *  \param[in]	aRight	Pointer to the right linked CBNodeBase
	 *  \param[in]	aExtraInt	Auxiliary \c int value
	 *  \attention Method generate \c std::runtime_error exception if parameters \p aLeft or \p aRight contain \c nullptr value.
	 *  \note Using of auxiliary \p ExtraInt value is useful for some projects only: multiset[same values counter], priority queue[priority]
	 */
		CBNode(const bool aValue, CBNode *aLeft, CBNode *aRight, int aExtraInt = 0): CBNodeBase(aLeft, aRight, aExtraInt), iValue(aValue)
			{}
		
	/** \brief Copy c'tor
	 *  \details Create new instance by copying only \p iValue parameter. All intrusive attributes (pointers,ExtraInt) are set to the their default values. 
	 *  \param[in]	aBNode	Original instance for copying
	 */
		CBNode(const CBNode& aBNode): CBNodeBase(), iValue(aBNode.Value())
			{}
		
	/** \brief String conversion c'tor
	 *  \details Create new instance from Value in the string. Pointer attributes are initialised to the \c this value.
	 *  \param[in] aStr Plain C string with value "0" or "1"
	 *  \param[in]	aExtraInt	Auxiliary \c int value
	 *  \note Using of auxiliary \p ExtraInt value is useful for some projects only: multiset[same values counter], priority queue[priority]
	 */
		CBNode(const char* aStr, int aExtraInt = 0): CBNodeBase(this, this, aExtraInt), iValue(false)
			{std::istringstream iss(aStr, std::istringstream::in); iss >> iValue;}
		
	/** \brief String conversion c'tor
	 *  \details Create new instance from Value in the string.
	 *  \param[in] aStr Plain C string with value int the format: "0" or "1"
	 *  \param[in]	aLeft	Pointer to the left linked CBNodeBase
	 *  \param[in]	aRight	Pointer to the right linked CBNodeBase
	 *  \param[in]	aExtraInt	Auxiliary \c int value
	 *  \attention Method generate \c std::runtime_error exception if parameters \p aLeft or \p aRight contain \c nullptr value.
	 *  \note Using of auxiliary \p ExtraInt value is useful for some projects only: multiset[same values counter], priority queue[priority]
	 */
		CBNode(const char* aStr, CBNode *aLeft, CBNode *aRight, int aExtraInt = 0): CBNodeBase(aLeft, aRight, aExtraInt), iValue(false)
			{std::istringstream iss(aStr, std::istringstream::in); iss >> iValue;}

	// d'tor
	/** \brief Virtual d'tor
	 */
		virtual ~CBNode()
			{iValue = false;}

	// setter/getter
	/** \brief Value setter
	 *  \param[in]	aValue	New Value
     */
		void SetValue(const bool aValue)
			{iValue = aValue;}
		
	/** \brief Value getter
	 *  \return Actual \c bool \p Value
	 */
		bool Value() const
			{return(iValue);}

	// operators
	/** \brief Complement operator
	 *  \return CBNode instance with complemented attribute Value. All intrusive attributes (pointers,ExtraInt) are set to the their default values.
	 */
		CBNode operator-() const
			{return(CBNode(!iValue));}
		
	/** \brief Assigment operator
	 *  \return CBNode instance with copied attribute Value. All intrusive attributes (pointers,ExtraInt) has previous values.
	 */
		CBNode& operator=(const CBNode& aBNode)
			{SetValue(aBNode.Value()); return(*this);}
		
	/** \brief Comparing by Value operator
	 *  \return Return \c bool result of comparation
	 */
		bool operator==(const CBNode& aBNode) const
			{return(iValue == aBNode.Value());}
		
	/** \brief Comparing by Value operator
	 *  \return Return \c bool result of comparation
	 */
		bool operator!=(const CBNode& aBNode) const
			{return(iValue != aBNode.Value());}
		
	/** \brief Comparing by Value operator
	 *  \return Return \c bool result of comparation
	 */
		bool operator<(const CBNode& aBNode) const
			{return(iValue < aBNode.Value());}
		
	/** \brief Comparing by Value operator
	 *  \return Return \c bool result of comparation
	 */
		bool operator>(const CBNode& aBNode) const
			{return(iValue > aBNode.Value());}


	/** \brief Output to the stream operator. (\em serialization)
	 *  \param[in]	aOStream	Output stream
	 *  \param[in]	aBNode	Serialized instantion of CBNode
	 *  \return Return \c std::ostream with serialized Value
	 */
		friend std::ostream& operator<<(std::ostream& aOStream, const CBNode& aBNode)
			{aOStream << aBNode.Value(); return aOStream;}
				
	/** \brief Input from the stream operator. (\em deserialization)
	 *  \param[in]	aIStream	Input stream
	 *  \param[out]	aBNode	Place for deserialized instantion of CBNode
	 *  \return Return rest of \c std::istream
	 */
		friend std::istream& operator>>(std::istream& aIStream, CBNode& aBNode)
			{aIStream >> aBNode.iValue; return aIStream;}

#ifdef CBNODE_TEST_VALUES
	// constants for testing CBNode functionality
	/** \brief First test value
	 *  \return Return \c bool value (\c false)
	 *  \note Useful for automated testing of CBNode functionality
	 */
		static bool TestValue0()
			{return(false);}
				
	/** \brief First test string value
	 *  \return Return string with \c bool value (\c false)
	 *  \note Useful for automated testing of CBNode functionality
	 */
		static const char* TestStringValue0()
			{return("0");}
				
	/** \brief Second test value
	 *  \return Return \c bool value (\c true)
	 *  \note Useful for automated testing of CBNode functionality
	 */
		static bool TestValue1()
			{return(true);}
				
	/** \brief Second test string value
	 *  \return Return string with \c bool value (\c true)
	 *  \note Useful for automated testing of CBNode functionality
	 */
		static const char* TestStringValue1()
			{return("1");}
				
#endif /* CBNODE_TEST_VALUES */

		}; /* class CBNode */

	} /* namespace CBNode_bool */

#endif /* __CBNODE_BOOL_H__ */
