#ifndef __CXITEMBASE_H__
#define __CXITEMBASE_H__
/** \file CXItemBase.h
 *  \brief CXItemBase base class header
 *  \details File contain CXItemBase definition.
 *  \warning Don't modify this file
 *  \author Pety 2014
 *  $Id: CXItemBase.h 112 2014-11-01 19:02:33Z petyovsky $
 */

#include <iostream>
#include <sstream>
#include <stdexcept>
#include <typeinfo>
#include "ClassInfo.h"
#include "demagle.h"
#include "check.h"

#if !( (__cplusplus >= 201103L) || (_MSC_VER >= 1600) || defined(nullptr) )
#define nullptr 0  // simulate C++11 nullptr feature
#define override  // disable C++11 override keyword 
#endif

// CXItem define config
#define CXITEM_TEST_VALUES		///< If defined CXItem have special values for testing
#define CXITEM_DTOR_EXCEPTIONS	///< If defined CXItem d'tor can generate exceptions
#ifndef RELEASE_BUILD
#define CXITEM_XPTR_CHECKING	///< If defined CXItem have special pointers for testing validity of pointers to items list
#endif /* RELEASE_BUILD */

/** \brief CXItemBase base class
 *  \details Definition of CXItemBase base class. There is defined all common methods and attributes.
 */
class CXItemBase
	{
	mutable std::ptrdiff_t iXPtr;		///< Differential (iXPtr = Right - Left) pointer to the both items (\em intrusive attribute)
#ifdef CXITEM_XPTR_CHECKING
	mutable const CXItemBase *iLeft;	///< Pointer to the left item (\em intrusive attribute) (Not accessible. Exist only in debug mode.)
	mutable const CXItemBase *iRight;	///< Pointer to the right item (\em intrusive attribute) (Not accessible. Exist only in debug mode.)
#endif /* CXITEM_XPTR_CHECKING */

	const ClassInfo<CXItemBase> iInstanceInfo;	///< Instance for class info statistics usage
	mutable int iExtraInt;				///< Auxiliary \c int value is useful for some projects only: multiset(for same value items counter], priority queue[for priority value] (\em intrusive attribute)

//c'tors
	CXItemBase(const CXItemBase& aItem); ///< Defined like a \c private for disallowing compiler implicit generation of copy c'tor

public:
/** \brief Implicit c'tor
 *  \details Pointer atributes are initialised to the \c this value others atributes contains zero.
 */
	CXItemBase(): iXPtr(0)
#ifdef CXITEM_XPTR_CHECKING
		, iLeft(this), iRight(this)
#endif /* CXITEM_XPTR_CHECKING */
		, iExtraInt(0)
		{}
	
/** \brief Conversion c'tor
 *  \param[in]	aLeft	Pointer to the left linked CXItemBase
 *  \param[in]	aRight	Pointer to the right linked CXItemBase
 *  \param[in]	aExtraInt	Auxiliary \c int value needed for project: multiset, priority queue...
 *  \attention Method generate \c std::invalid_argument exception if parameters \p aLeft or \p aRight contain \c nullptr value.
 *  \note Using of auxiliary \p ExtraInt value is useful for some projects only: multiset[same values counter], priority queue[priority]
 */
	CXItemBase(CXItemBase *aLeft, CXItemBase *aRight, int aExtraInt = 0): iXPtr(0)
#ifdef CXITEM_XPTR_CHECKING
		, iLeft(this), iRight(this)
#endif /* CXITEM_XPTR_CHECKING */
		, iExtraInt(aExtraInt)
		{SetLeftRight(aLeft, aRight);}

//d'tor
/** \brief Virtual d'tor
 *  \attention Method check and generate \c std::runtime_error exception if instance is part of the any list! So both pointers calculated from \c iXPtr must point to the \c this value before you call d'tor.
 *  \note Generating exceptions from d'tors is problematic practice (for example during early development process), because can generate multiple exceptions in the same moment. So, throwing of \c std::runtime_error exception can be supress be undefining preprocessor symbol \c CXItem_DTOR_EXCEPTIONS, but error message is always reported to the \c std::cerr.
 */
	virtual ~CXItemBase()
		{
		std::ostringstream oss;
		if((iXPtr)
#ifdef CXITEM_XPTR_CHECKING
		   || (iLeft != this) || (iRight != this)
#endif /* CXITEM_XPTR_CHECKING */
		   )
			{
			oss << "Item is part of the some list structure. Instance of class: \'" << DM(typeid(*this).name()) << '#' << ID() << "\' must be unlinked from list first before deleting!!";

#ifdef CXITEM_DTOR_EXCEPTIONS
			throw std::runtime_error(oss.str());
#else
			std::cerr << std::endl << "ERROR: " << oss.str() << std::endl << '\a';
#endif
			}
		}


// intrusive setters/getters
/** \brief Left pointer setter
 *  \param[in]	aLeftNew	Pointer to the new left linked CXItemBase
 *  \param[in]	aLeftOld	Pointer to the old left linked CXItemBase
 *  \attention Method generate \c std::invalid_argument exception if parameters \p aLeftNew or \p aLeftOld contains \c nullptr value.
 *  \attention Method (only in debug mode) generate \c std::runtime_error exception if parameters \p aLeftOld is differ from stored \c iLeft value.
 */
	void SetLeft(const CXItemBase *aLeftNew, const CXItemBase *aLeftOld) const
		{
		if(!aLeftNew || !aLeftOld)
			{
			std::ostringstream oss;
			oss << "Parameters is nullptr value. Instance of class: \'" << DM(typeid(*this).name()) << '#' << ID() << "\'!!";
			throw std::invalid_argument(oss.str());
			}
#ifdef CXITEM_XPTR_CHECKING
		if(aLeftOld != iLeft)
			{
			std::ostringstream oss;
			oss << "Wrong aLeftOld parameter value. Instance of class: \'" << DM(typeid(*this).name()) << '#' << ID() << "\'!!";
			throw std::runtime_error(oss.str());
			}
		iLeft = aLeftNew;
#endif /* CXITEM_XPTR_CHECKING */

		iXPtr += reinterpret_cast<const char*>(aLeftOld) - reinterpret_cast<const char*>(aLeftNew); // (iXPtr = Right - Left)
		}

/** \brief Left pointer getter
 *  \param[in]	aRight	Pointer to the right linked CXItemBase
 *  \return Actual \p left pointer value
 *  \attention Method generate \c std::invalid_argument exception if parameters \p aRight contain \c nullptr value.
 *  \attention Method (only in debug mode) generate \c std::runtime_error exception if parameters \p aRight is differ from stored \c iRight value.
 *  \attention Method (only in debug mode) generate \c std::runtime_error exception if return Left value is differ from stored \c iLeft value.
 */
	const CXItemBase* Left(const CXItemBase* aRight) const
		{
		if(!aRight)
			{
			std::ostringstream oss; oss << "Invalid aRight parameter (nullptr) for method Left() of instance of class: \'" << DM(typeid(*this).name()) << '#' << ID() << "\'";
			throw std::invalid_argument(oss.str());
			}
#ifdef CXITEM_XPTR_CHECKING
		if(aRight != iRight)
			{
			std::ostringstream oss;
			oss << "Invalid aRight parameter value for method Left() of instance of class: \'" << DM(typeid(*this).name()) << '#' << ID() << "\'!!";
			throw std::runtime_error(oss.str());
			}
#endif /* CXITEM_XPTR_CHECKING */

		const CXItemBase *left(reinterpret_cast<const CXItemBase *>(reinterpret_cast<const char*>(aRight) - iXPtr)); // (iXPtr = Right - Left)

		if(!left)
			{
			std::ostringstream oss;
			oss << "Calculated Left pointer is nullptr value. Instance of class: \'" << DM(typeid(*this).name()) << '#' << ID() << "\'!!";
			throw std::invalid_argument(oss.str());
			}

#ifdef CXITEM_XPTR_CHECKING
		if(left != iLeft)
			{
			std::ostringstream oss;
			oss << "Calculated Left pointer is different from stored. Instance of class: \'" << DM(typeid(*this).name()) << '#' << ID() << "\' lose data consistency!!";
			throw std::runtime_error(oss.str());
			}
#endif /* CXITEM_XPTR_CHECKING */

		return(left);
		}

/** \brief Right pointer setter
 *  \param[in]	aRightNew	Pointer to the new right linked CXItemBase
 *  \param[in]	aRightOld	Pointer to the old right linked CXItemBase
 *  \attention Method generate \c std::invalid_argument exception if parameters \p aRightNew or \p aRightOld contain \c nullptr value.
 *  \attention Method (only in debug mode) generate \c std::runtime_error exception if parameters \p aRightOld is differ from stored \c iRight value.
 */
	void SetRight(const CXItemBase *aRightNew, const CXItemBase *aRightOld) const
		{
		if(!aRightNew || !aRightOld)
			{
			std::ostringstream oss;
			oss << "Parameters is nullptr value. Instance of class: \'" << DM(typeid(*this).name()) << '#' << ID() << "\'!!";
			throw std::invalid_argument(oss.str());
			}
#ifdef CXITEM_XPTR_CHECKING
		if(aRightOld != iRight)
			{
			std::ostringstream oss;
			oss << "Wrong aLeftOld parameter value. Instance of class: \'" << DM(typeid(*this).name()) << '#' << ID() << "\'!!";
			throw std::runtime_error(oss.str());
			}
		iRight = aRightNew;
#endif /* CXITEM_XPTR_CHECKING */

		iXPtr += reinterpret_cast<const char*>(aRightNew) - reinterpret_cast<const char*>(aRightOld); // (iXPtr = Right - Left)
		}

/** \brief Right pointer getter
 *  \param[in]	aLeft	Pointer to the left linked CXItemBase
 *  \return Actual \p right pointer value
 *  \attention Method generate \c std::invalid_argument exception if parameters \p aLeft contain \c nullptr value.
 *  \attention Method (only in debug mode) generate \c std::runtime_error exception if parameters \p aLeft is differ from stored \c iLeft value.
 *  \attention Method (only in debug mode) generate \c std::runtime_error exception if return Right value is differ from stored \c iRight value.
 */
	const CXItemBase* Right(const CXItemBase* aLeft) const
		{
		if(!aLeft)
			{
			std::ostringstream oss; oss << "Invalid aLeft parameter (nullptr) of instance of class: \'" << DM(typeid(*this).name()) << '#' << ID() << "\'";
			throw std::invalid_argument(oss.str());
			}
#ifdef CXITEM_XPTR_CHECKING
		if(aLeft != iLeft)
			{
			std::ostringstream oss;
			oss << "Wrong aLeft parameter value. Instance of class: \'" << DM(typeid(*this).name()) << '#' << ID() << "\'!!";
			throw std::runtime_error(oss.str());
			}
#endif /* CXITEM_XPTR_CHECKING */

		const CXItemBase *right(reinterpret_cast<const CXItemBase *>(iXPtr + reinterpret_cast<const char*>(aLeft))); // (iXPtr = Right - Left)

		if(!right)
			{
			std::ostringstream oss;
			oss << "Calculated Right pointer is nullptr value. Instance of class: \'" << DM(typeid(*this).name()) << '#' << ID() << "\'!!";
			throw std::invalid_argument(oss.str());
			}

#ifdef CXITEM_XPTR_CHECKING
		if(right != iRight)
			{
			std::ostringstream oss;
			oss << "Calculated Right pointer is different from stored. Instance of class: \'" << DM(typeid(*this).name()) << '#' << ID() << "\' lose data consistency!!";
			throw std::runtime_error(oss.str());
			}
#endif /* CXITEM_XPTR_CHECKING */

		return(right);
		}

/** \brief Combined Left and Right pointers setter
 *  \param[in]	aLeft	Pointer to the new left linked CXItemBase
 *  \param[in]	aRight	Pointer to the new right linked CXItemBase
 *  \attention Method generate \c std::invalid_argument exception if parameters \p aLeft or \p aRight contain \c nullptr value.
 *  \attention Method check and generate \c std::runtime_error exception if instance is part of the any list! So both pointers calculated from \c iXPtr must point to the \c this value before you call this method.
 */
	void SetLeftRight(const CXItemBase *aLeft, const CXItemBase *aRight) const
		{
		if(!aLeft)
			{
			std::ostringstream oss; oss << "Invalid aLeft parameter (nullptr) of instance of class: \'" << DM(typeid(*this).name()) << '#' <<  ID() << "\'";
			throw std::invalid_argument(oss.str());
			}
		if(!aRight)
			{
			std::ostringstream oss; oss << "Invalid aRight parameter (nullptr) of instance of class: \'" << DM(typeid(*this).name()) << '#' << ID() << "\'";
			throw std::invalid_argument(oss.str());
			}

		if( (iXPtr)
#ifdef CXITEM_XPTR_CHECKING
		   || (iLeft != this) || (iRight != this)
#endif /* CXITEM_XPTR_CHECKING */
		  )
			{
			std::ostringstream oss;
			oss << "Item is part of the some list structure. Instance of class: \'" << DM(typeid(*this).name()) << '#' << ID() << "\' must be unlinked from list first!!";
			throw std::runtime_error(oss.str());
			}
#ifdef CXITEM_XPTR_CHECKING
		iRight = aRight;
		iLeft = aLeft;
#endif /* CXITEM_XPTR_CHECKING */

		iXPtr = reinterpret_cast<const char*>(aRight) - reinterpret_cast<const char*>(aLeft); // (iXPtr = Right - Left)
		}


// intrusive setter/getter, useful for some projects only: multiset, priority queue
/** \brief Auxiliary ExtraInt value setter
 *  \param[in]	aExtraInt	New ExtraInt value
 *  \note Using of auxiliary \p ExtraInt value is useful for some projects only: multiset[same values counter], priority queue[priority]
 */
	void SetExtraInt(const int aExtraInt) const
		{iExtraInt = aExtraInt;}
	
/** \brief Auxiliary ExtraInt value getter
 *  \return Actual \p ExtraInt value
 *  \note Using of auxiliary \p ExtraInt value is useful for some projects only: multiset[same values counter], priority queue[priority]
 */
	int ExtraInt() const
		{return(iExtraInt);}

// InstanceInfo getters
/** \brief ID getter
 *  \return Unique instance ID
 */
	virtual unsigned ID() const
		{return(iInstanceInfo.ID());}

	}; /* class CXItemBase */

#endif /* __CXITEMBASE_H__ */
