#ifndef __CXITEM_BOOL_H__
#define __CXITEM_BOOL_H__
/** \file CXItem_bool.h
 *  \brief CXItem_bool class header
 *  \details File contain CXItem_bool::CXItem definition.
 *  \warning Don't modify this file
 *  \author Pety 2014
 *  $Id: CXItem_bool.h 112 2014-11-01 19:02:33Z petyovsky $
 */

#include "CXItemBase.h"
#include "check.h"

/** \brief Namespace for encapsulating of \c bool variant of CXItem class
 *  \details For selecting this variant of CXItem class uncomment \c using \c namespace section in the CXItem.h
 */
namespace CXItem_bool
	{
	/** \brief CXItem class (\c bool variant)
	 *  \details Definition of CXItem class (\c bool variant). There is defined all common methods and attributes.
	 */
	class CXItem: public CXItemBase
		{
		const ClassInfo<CXItem> iInstanceInfo;	///< Instance for class info statistics usage
		bool iValue; ///< Encapsulated \c bool value

	public:
	// c'tors
	/** \brief Implicit c'tor
	 *  \details Value attributes is set to \c false. Inherited attributes are initialised by CXItemBase implicit c'tor. 
	 */
		CXItem(): iValue(false)
			{}
		
	/** \brief Conversion c'tor
	 *  \details Pointer attributes are initialised to the \c this value.
   	 *  \param[in]	aValue New encapsulated \c bool Value
	 *  \param[in]	aExtraInt	Auxiliary \c int value
	 *  \note Using of auxiliary \p ExtraInt value is useful for some projects only: multiset[same values counter], priority queue[priority]
	 */
		explicit CXItem(const bool aValue, int aExtraInt = 0): CXItemBase(this, this, aExtraInt), iValue(aValue)
			{}
		
	/** \brief Conversion c'tor
   	 *  \param[in]	aValue New encapsulated \c bool Value
	 *  \param[in]	aLeft	Pointer to the left linked CXItemBase
	 *  \param[in]	aRight	Pointer to the right linked CXItemBase
	 *  \param[in]	aExtraInt	Auxiliary \c int value
	 *  \attention Method generate \c std::invalid_argument exception if parameters \p aLeft or \p aRight contain \c nullptr value.
	 *  \note Using of auxiliary \p ExtraInt value is useful for some projects only: multiset[same values counter], priority queue[priority]
	 */
		CXItem(const bool aValue, CXItem *aLeft, CXItem *aRight, int aExtraInt = 0): CXItemBase(aLeft, aRight, aExtraInt), iValue(aValue)
			{}
		
	/** \brief Copy c'tor
	 *  \details Create new instance by copying only \p iValue parameter. All intrusive attributes (pointers,ExtraInt) are set to the their default values. 
	 *  \param[in]	aItem	Original instance for copying
	 */
		CXItem(const CXItem& aItem): CXItemBase(), iValue(aItem.Value())
			{}
		
	/** \brief String conversion c'tor
	 *  \details Create new instance from Value in the string. Pointer attributes are initialised to the \c this value.
	 *  \param[in] aStr Plain C string with value "0" or "1"
	 *  \param[in]	aExtraInt	Auxiliary \c int value
	 *  \note Using of auxiliary \p ExtraInt value is useful for some projects only: multiset[same values counter], priority queue[priority]
	 */
		explicit CXItem(const char* aStr, int aExtraInt = 0): CXItemBase(this, this, aExtraInt), iValue(false)
			{std::istringstream iss(aStr, std::istringstream::in); iss >> std::boolalpha >> iValue;}
		
	/** \brief String conversion c'tor
	 *  \details Create new instance from Value in the string.
	 *  \param[in] aStr Plain C string with value int the format: "0" or "1"
	 *  \param[in]	aLeft	Pointer to the left linked CXItemBase
	 *  \param[in]	aRight	Pointer to the right linked CXItemBase
	 *  \param[in]	aExtraInt	Auxiliary \c int value
	 *  \attention Method generate \c std::invalid_argument exception if parameters \p aLeft or \p aRight contain \c nullptr value.
	 *  \note Using of auxiliary \p ExtraInt value is useful for some projects only: multiset[same values counter], priority queue[priority]
	 */
		CXItem(const char* aStr, CXItem *aLeft, CXItem *aRight, int aExtraInt = 0): CXItemBase(aLeft, aRight, aExtraInt), iValue(false)
			{std::istringstream iss(aStr, std::istringstream::in); iss >> std::boolalpha >> iValue;}

	// d'tor
	/** \brief Virtual d'tor
	 */
		virtual ~CXItem()
			{iValue = false;}

	// setter/getter
	/** \brief Value setter
	 *  \param[in]	aValue	New Value
     */
		void SetValue(const bool aValue)
			{iValue = aValue;}
		
	/** \brief Value getter
	 *  \return Actual \c bool \p Value
	 */
		bool Value() const
			{return(iValue);}

	// operators
	/** \brief Complement operator
	 *  \return CXItem instance with complemented attribute Value. All intrusive attributes (pointers,ExtraInt) are set to the their default values.
	 */
		CXItem operator-() const
			{return(CXItem(!iValue));}
		
	/** \brief Assigment operator
	 *  \return CXItem instance with copied attribute Value. All intrusive attributes (pointers,ExtraInt) has previous values.
	 */
		CXItem& operator=(const CXItem& aItem)
			{SetValue(aItem.Value()); return(*this);}
		
	/** \brief Comparing by Value operator
	 *  \return Return \c bool result of comparation
	 */
		bool operator==(const CXItem& aItem) const
			{return(iValue == aItem.Value());}
		
	/** \brief Comparing by Value operator
	 *  \return Return \c bool result of comparation
	 */
		bool operator!=(const CXItem& aItem) const
			{return(iValue != aItem.Value());}
		
	/** \brief Comparing by Value operator
	 *  \return Return \c bool result of comparation
	 */
		bool operator<(const CXItem& aItem) const
			{return(iValue < aItem.Value());}
		
	/** \brief Comparing by Value operator
	 *  \return Return \c bool result of comparation
	 */
		bool operator>(const CXItem& aItem) const
			{return(iValue > aItem.Value());}


	/** \brief Output to the stream operator. (\em serialization)
	 *  \param[in]	aOStream	Output stream
	 *  \param[in]	aItem	Serialized instantion of CXItem
	 *  \return Return \c std::ostream with serialized Value
	 */
		friend std::ostream& operator<<(std::ostream& aOStream, const CXItem& aItem)
			{aOStream << std::boolalpha << aItem.Value(); return aOStream;}
				
	/** \brief Input from the stream operator. (\em deserialization)
	 *  \param[in]	aIStream	Input stream
	 *  \param[out]	aItem	Place for deserialized instantion of CXItem
	 *  \return Return rest of \c std::istream
	 */
		friend std::istream& operator>>(std::istream& aIStream, CXItem& aItem)
			{aIStream >> std::boolalpha >> aItem.iValue; return aIStream;}

#ifdef CXITEM_TEST_VALUES
		// constants for testing CXItem functionality

		/** \brief First test value
		*  \return Return \c bool value (\c false)
		*  \note Useful for automated testing of CXItem functionality
		*/
		static bool TestValue0()
			{return(false);}

		/** \brief First test string value
		*  \return Return string with \c bool value (\c false)
		*  \note Useful for automated testing of CXItem functionality
		*/
		static const char* TestStringValue0()
			{return("false");}

		/** \brief Second test value
		*  \return Return \c bool value (\c true)
		*  \note Useful for automated testing of CXItem functionality
		*/
		static bool TestValue1()
			{return(true);}

		/** \brief Second test string value
		*  \return Return string with \c bool value (\c true)
		*  \note Useful for automated testing of CXItem functionality
		*/
		static const char* TestStringValue1()
			{return("true");}

		/** \brief Random test value
		*  \return Return random \c bool value
		*  \note Useful for automated testing of CXItem functionality
		*/
		static bool TestValueRandom()
			{return((std::rand() % 2) != 0);}

		/** \brief Random test string value
		*  \return Return string with random \c bool value
		*  \note Useful for automated testing of CXItem functionality
		*/
		static const char* TestStringValueRandom()
			{return(TestValueRandom() ? TestStringValue1() : TestStringValue0());}

#endif /* CXITEM_TEST_VALUES */

	// InstanceInfo getters
	/** \brief ID getter
	*  \return Unique instance ID
	*/
		virtual unsigned ID() const override
			{return(iInstanceInfo.ID());}

		}; /* class CXItem */

	} /* namespace CXItem_bool */

#endif /* __CXITEM_BOOL_H__ */
